#include <iostream>
#include <vector>
#include <algorithm>
#include <cmath>
using namespace std;

struct Point {
    long long x, y;
};

bool cmp(const Point& a, const Point& b) {
    return a.x < b.x || (a.x == b.x && a.y < b.y);
}

long long cross(const Point& O, const Point& A, const Point& B) {
    return (A.x - O.x) * (B.y - O.y) - (A.y - O.y) * (B.x - O.x);
}

vector<Point> convex_hull(vector<Point> points) {
    int n = points.size();
    if (n <= 2) return points;
    
    sort(points.begin(), points.end(), cmp);
    
    vector<Point> hull;
    
    for (int i = 0; i < n; i++) {
        while (hull.size() >= 2 && cross(hull[hull.size()-2], hull[hull.size()-1], points[i]) <= 0)
            hull.pop_back();
        hull.push_back(points[i]);
    }
    
    int lower_size = hull.size();
    for (int i = n - 2; i >= 0; i--) {
        while (hull.size() > lower_size && cross(hull[hull.size()-2], hull[hull.size()-1], points[i]) <= 0)
            hull.pop_back();
        hull.push_back(points[i]);
    }
    
    hull.pop_back();
    return hull;
}

long long gcd(long long a, long long b) {
    a = abs(a); b = abs(b);
    while (b) {
        long long t = b;
        b = a % b;
        a = t;
    }
    return a;
}

int main() {
    ios_base::sync_with_stdio(false);
    cin.tie(NULL);
    
    int n;
    cin >> n;
    
    vector<Point> points(n);
    for (int i = 0; i < n; i++) {
        cin >> points[i].x >> points[i].y;
    }
    
    vector<Point> hull = convex_hull(points);
    int h = hull.size();
    
    if (h <= 1) {
        cout << "1 0 0\n1 0 0\n";
        return 0;
    }
    
    double max_width = -1;
    long long best_a1, best_b1, best_c1;
    long long best_a2, best_b2, best_c2;
    
    // Проверяем направления рёбер выпуклой оболочки
    for (int i = 0; i < h; i++) {
        int j = (i + 1) % h;
        
        long long dx = hull[j].x - hull[i].x;
        long long dy = hull[j].y - hull[i].y;
        
        // Проверяем два направления: перпендикуляр к ребру и само ребро
        long long normals[2][2] = {{-dy, dx}, {dx, dy}};
        
        for (int k = 0; k < 2; k++) {
            long long a = normals[k][0];
            long long b = normals[k][1];
            
            if (a == 0 && b == 0) continue;
            
            long long min_val = LLONG_MAX;
            long long max_val = LLONG_MIN;
            
            for (const auto& p : points) {
                long long val = a * p.x + b * p.y;
                min_val = min(min_val, val);
                max_val = max(max_val, val);
            }
            
            double width = (max_val - min_val) / sqrt((double)(a * a + b * b));
            
            if (width > max_width) {
                max_width = width;
                long long c1 = -min_val;
                long long c2 = -max_val;
                
                long long g1 = gcd(gcd(abs(a), abs(b)), abs(c1));
                if (g1 == 0) g1 = 1;
                best_a1 = a / g1;
                best_b1 = b / g1;
                best_c1 = c1 / g1;
                
                long long g2 = gcd(gcd(abs(a), abs(b)), abs(c2));
                if (g2 == 0) g2 = 1;
                best_a2 = a / g2;
                best_b2 = b / g2;
                best_c2 = c2 / g2;
            }
        }
    }
    
    cout << best_a1 << " " << best_b1 << " " << best_c1 << "\n";
    cout << best_a2 << " " << best_b2 << " " << best_c2 << "\n";
    
    return 0;
}
