﻿#include <iostream>
#include <vector>
#include <algorithm>
#include <string>
#include <stdexcept>
#include <limits>
#include <unordered_set>
using namespace std;

struct BigInt {
    vector<int> digits;
    bool is_negative;

    BigInt() : digits({ 0 }), is_negative(false) {}
    BigInt(const string& s) { *this = from_string(s); }
    BigInt(long long n) { *this = from_long_long(n); }

    void removeLeadingZeros() {
        while (digits.size() > 1 && digits.back() == 0) {
            digits.pop_back();
        }
        if (digits.size() == 1 && digits[0] == 0) {
            is_negative = false;
        }
    }

    BigInt from_string(const string& s) {
        BigInt result;
        result.digits.clear();

        int start = 0;
        if (s[0] == '-') {
            result.is_negative = true;
            start = 1;
        }
        else {
            result.is_negative = false;
        }

        for (int i = s.size() - 1; i >= start; --i) {
            if (s[i] < '0' || s[i] > '9') {
                throw invalid_argument("Invalid character in number string");
            }
            result.digits.push_back(s[i] - '0');
        }

        result.removeLeadingZeros();
        return result;
    }

    BigInt from_long_long(long long n) {
        BigInt result;
        result.digits.clear();
        result.is_negative = (n < 0);

        unsigned long long num;
        if (n < 0) {
            if (n == numeric_limits<long long>::min()) {
                num = static_cast<unsigned long long>(numeric_limits<long long>::max()) + 1;
            }
            else {
                num = static_cast<unsigned long long>(-n);
            }
        }
        else {
            num = static_cast<unsigned long long>(n);
        }

        if (num == 0) {
            result.digits.push_back(0);
        }
        else {
            while (num > 0) {
                result.digits.push_back(num % 10);
                num /= 10;
            }
        }

        return result;
    }

    long long to_long_long(BigInt value)
    {
        long long res=0;
        long long mnoj=1;
        for (int i = 0; i < value.digits.size(); i++)
        {
            res += value.digits[i] * mnoj;
            mnoj *= 10;
        }
        if (value.is_negative)
            return -res;
        else
            return res;
    }
    string to_string() const {
        string result;
        if (is_negative && !(digits.size() == 1 && digits[0] == 0)) {
            result += '-';
        }
        for (int i = digits.size() - 1; i >= 0; --i) {
            result += char(digits[i] + '0');
        }
        return result;
    }

    int compare_abs(const BigInt& other) const {
        if (digits.size() != other.digits.size()) {
            return digits.size() > other.digits.size() ? 1 : -1;
        }
        for (int i = digits.size() - 1; i >= 0; --i) {
            if (digits[i] != other.digits[i]) {
                return digits[i] > other.digits[i] ? 1 : -1;
            }
        }
        return 0;
    }

    int compare(const BigInt& other) const {
        if (is_negative && !other.is_negative) return -1;
        if (!is_negative && other.is_negative) return 1;

        int abs_comp = compare_abs(other);
        return is_negative ? -abs_comp : abs_comp;
    }

    bool operator==(const BigInt& other) const { return compare(other) == 0; }
    bool operator!=(const BigInt& other) const { return compare(other) != 0; }
    bool operator<(const BigInt& other) const { return compare(other) < 0; }
    bool operator>(const BigInt& other) const { return compare(other) > 0; }
    bool operator<=(const BigInt& other) const { return compare(other) <= 0; }
    bool operator>=(const BigInt& other) const { return compare(other) >= 0; }

    BigInt operator-() const {
        BigInt result = *this;
        if (!(digits.size() == 1 && digits[0] == 0)) {
            result.is_negative = !is_negative;
        }
        return result;
    }

    BigInt operator+(const BigInt& other) const {
        if (is_negative == other.is_negative) {
            BigInt result;
            result.digits.clear();
            result.is_negative = is_negative;

            int carry = 0;
            size_t max_size = max(digits.size(), other.digits.size());

            for (size_t i = 0; i < max_size || carry; ++i) {
                int sum = carry;
                if (i < digits.size()) sum += digits[i];
                if (i < other.digits.size()) sum += other.digits[i];

                result.digits.push_back(sum % 10);
                carry = sum / 10;
            }

            result.removeLeadingZeros();
            return result;
        }
        else {
            if (compare_abs(other) >= 0) {
                BigInt result = *this;
                result.is_negative = is_negative;

                int borrow = 0;
                for (size_t i = 0; i < other.digits.size() || borrow; ++i) {
                    int diff = digits[i] - borrow;
                    if (i < other.digits.size()) diff -= other.digits[i];

                    if (diff < 0) {
                        diff += 10;
                        borrow = 1;
                    }
                    else {
                        borrow = 0;
                    }

                    if (i < result.digits.size()) {
                        result.digits[i] = diff;
                    }
                    else {
                        result.digits.push_back(diff);
                    }
                }

                result.removeLeadingZeros();
                return result;
            }
            else {
                return other + *this;
            }
        }
    }

    BigInt operator-(const BigInt& other) const {
        return *this + (-other);
    }

    BigInt operator*(const BigInt& other) const {
        BigInt result;
        result.digits.resize(digits.size() + other.digits.size(), 0);
        result.is_negative = is_negative != other.is_negative;

        for (size_t i = 0; i < digits.size(); ++i) {
            int carry = 0;
            for (size_t j = 0; j < other.digits.size() || carry; ++j) {
                long long product = result.digits[i + j] +
                    digits[i] * (j < other.digits.size() ? other.digits[j] : 0) +
                    carry;
                result.digits[i + j] = product % 10;
                carry = product / 10;
            }
        }

        result.removeLeadingZeros();
        return result;
    }

    BigInt operator/(const BigInt& other) const {
        if (other.digits.size() == 1 && other.digits[0] == 0) {
            throw runtime_error("Division by zero");
        }

        BigInt result, current;
        result.digits.clear();
        result.is_negative = is_negative != other.is_negative;

        BigInt abs_this = *this;
        abs_this.is_negative = false;
        BigInt abs_other = other;
        abs_other.is_negative = false;

        for (int i = abs_this.digits.size() - 1; i >= 0; --i) {
            current.digits.insert(current.digits.begin(), abs_this.digits[i]);
            current.removeLeadingZeros();

            int count = 0;
            while (current.compare_abs(abs_other) >= 0) {
                current = current - abs_other;
                count++;
            }

            result.digits.push_back(count);
        }

        reverse(result.digits.begin(), result.digits.end());
        result.removeLeadingZeros();
        return result;
    }

    BigInt operator%(const BigInt& other) const {
        BigInt quotient = *this / other;
        return *this - quotient * other;
    }

    friend ostream& operator<<(ostream& os, const BigInt& num) {
        os << num.to_string();
        return os;
    }

};
BigInt pow(const BigInt& base, const BigInt& exponent) {
    BigInt result(1);
    BigInt b = base;
    BigInt e = exponent;
    BigInt zero(0);
    BigInt one(1);

    while (e > zero) {
        if ((e % BigInt(2)) == one) {
            result = result * b;
        }
        b = b * b;
        e = e / BigInt(2);
    }
    return result;
}

int main() {
    int n;
    std::cin >> n;

    long long vl;
    std::cin >> vl;

    BigInt M(999983);
    long long V = 0;
    long long G = 0;

    for (int i = 0; i < n; i++)
    {
        BigInt Bvl(vl);
        Bvl = (Bvl * 618023 + 1) % M;
        vl = Bvl.to_long_long(Bvl);
        if (((G >> vl) & 1) == 0)
        {
            long long cur = V;
            G ^= (cur << vl);
            G ^= (1LL << (2 * vl));
            V |= (1LL << vl);
        }
    }

    int c = 0;
    for (long long tmp = V; tmp != 0; tmp &= (tmp - 1)) c++;

    std::cout << c << std::endl;

    return 0;
}

