#include <iostream>
#include <vector>
#include <numeric>
#include <algorithm>

namespace NTT {
    const int MOD = 998244353;
    const int ROOT = 3;
    const int MAX_N = 1 << 21;

    long long power(long long base, long long exp) {
        long long res = 1;
        base %= MOD;
        while (exp > 0) {
            if (exp % 2 == 1) res = (res * base) % MOD;
            base = (base * base) % MOD;
            exp /= 2;
        }
        return res;
    }

    long long modInverse(long long n) {
        return power(n, MOD - 2);
    }

    void fft(std::vector<long long>& a, bool invert) {
        int n = a.size();

        for (int i = 1, j = 0; i < n; i++) {
            int bit = n >> 1;
            for (; j & bit; bit >>= 1)
                j ^= bit;
            j ^= bit;
            if (i < j)
                std::swap(a[i], a[j]);
        }

        for (int len = 2; len <= n; len <<= 1) {
            long long wlen = power(ROOT, (MOD - 1) / len);
            if (invert)
                wlen = modInverse(wlen);
            for (int i = 0; i < n; i += len) {
                long long w = 1;
                for (int j = 0; j < len / 2; j++) {
                    long long u = a[i + j], v = (a[i + j + len / 2] * w) % MOD;
                    a[i + j] = (u + v) % MOD;
                    a[i + j + len / 2] = (u - v + MOD) % MOD;
                    w = (w * wlen) % MOD;
                }
            }
        }

        if (invert) {
            long long n_inv = modInverse(n);
            for (long long& x : a)
                x = (x * n_inv) % MOD;
        }
    }

    std::vector<long long> multiply(std::vector<long long>& a, std::vector<long long>& b) {
        int sz = 1;
        while (sz < a.size() + b.size()) sz <<= 1;
        a.resize(sz);
        b.resize(sz);

        fft(a, false);
        fft(b, false);

        std::vector<long long> c(sz);
        for (int i = 0; i < sz; i++)
            c[i] = (a[i] * b[i]) % MOD;

        fft(c, true);
        return c;
    }
}

const int MOD_S = 999983;
const int BLOCK_SIZE = 256;
const int NTT_SIZE = 1 << 21;

std::vector<bool> V_main(MOD_S, false);
std::vector<bool> P_main(2 * MOD_S, false);
std::vector<int> V_buffer;
std::vector<bool> P_buffer(2 * MOD_S, false);
std::vector<bool> exists_in_buffer(MOD_S, false);
int V_main_size = 0;

void rebuild() {
    if (V_buffer.empty()) return;

    for (int x : V_buffer) {
        if (!V_main[x]) {
            V_main[x] = true;
            V_main_size++;
        }
    }
    V_buffer.clear();
    std::fill(P_buffer.begin(), P_buffer.end(), false);
    std::fill(exists_in_buffer.begin(), exists_in_buffer.end(), false);

    std::vector<long long> V_poly(MOD_S, 0);
    for (int i = 0; i < MOD_S; ++i) {
        if (V_main[i]) {
            V_poly[i] = 1;
        }
    }

    auto Sq_poly = NTT::multiply(V_poly, V_poly);
    Sq_poly.resize(2 * MOD_S);

    for (int s = 0; s < 2 * MOD_S; ++s) {
        long long Ts = Sq_poly[s];
        long long Ds = (s % 2 == 0 && s / 2 < MOD_S && V_main[s / 2]) ? 1 : 0;
        P_main[s] = (((Ts + Ds) / 2) % 2) != 0;
    }
}

int main() {
    int n;
    long long s_val;
    std::cin >> n >> s_val;

    for (int i = 0; i < n; ++i) {
        if (i > 0 && i % BLOCK_SIZE == 0) {
            rebuild();
        }

        s_val = (s_val * 618023 + 1) % MOD_S;

        bool parity = P_main[s_val];
        for (int x : V_buffer) {
            if (s_val >= x && V_main[s_val - x]) {
                parity = !parity;
            }
        }
        if (P_buffer[s_val]) {
            parity = !parity;
        }

        if (!parity) {
            if (!V_main[s_val] && !exists_in_buffer[s_val]) {
                for (int x : V_buffer) {
                    P_buffer[s_val + x] = !P_buffer[s_val + x];
                }
                P_buffer[2 * s_val] = !P_buffer[2 * s_val];
                V_buffer.push_back(s_val);
                exists_in_buffer[s_val] = true;
            }
        }
    }

    std::cout << V_main_size + V_buffer.size() << std::endl;

    return 0;
}