#include <algorithm>
#include <climits>
#include <iostream>
#include <vector>

typedef long long ll;

struct Point {
  ll x, y;
};

ll cross_product(Point p1, Point p2, Point p3) {
  return (p2.x - p1.x) * (p3.y - p1.y) - (p2.y - p1.y) * (p3.x - p1.x);
}

std::vector<Point> build_convex_hull(std::vector<Point> &points) {
  int n = points.size();
  if (n <= 2) {
    return points;
  }

  std::sort(points.begin(), points.end(), [](Point a, Point b) {
    if (a.x != b.x)
      return a.x < b.x;
    return a.y < b.y;
  });

  std::vector<Point> lower_hull, upper_hull;

  for (const auto &p : points) {
    while (lower_hull.size() >= 2 &&
           cross_product(lower_hull[lower_hull.size() - 2], lower_hull.back(),
                         p) <= 0) {
      lower_hull.pop_back();
    }
    lower_hull.push_back(p);
  }

  for (int i = n - 1; i >= 0; --i) {
    const auto &p = points[i];
    while (upper_hull.size() >= 2 &&
           cross_product(upper_hull[upper_hull.size() - 2], upper_hull.back(),
                         p) <= 0) {
      upper_hull.pop_back();
    }
    upper_hull.push_back(p);
  }

  lower_hull.pop_back();
  upper_hull.pop_back();
  lower_hull.insert(lower_hull.end(), upper_hull.begin(), upper_hull.end());

  return lower_hull;
}

int main() {
  std::ios_base::sync_with_stdio(false);
  std::cin.tie(NULL);

  int n;
  std::cin >> n;

  std::vector<Point> points(n);
  for (int i = 0; i < n; ++i) {
    std::cin >> points[i].x >> points[i].y;
  }

  if (n == 2) {
    ll a = points[1].x - points[0].x;
    ll b = points[1].y - points[0].y;
    ll c1 = -a * points[0].x - b * points[0].y;
    ll c2 = -a * points[1].x - b * points[1].y;
    std::cout << a << " " << b << " " << c1 << "\n";
    std::cout << a << " " << b << " " << c2 << "\n";
    return 0;
  }

  std::vector<Point> hull = build_convex_hull(points);

  ll best_a = 0, best_b = 0, best_c1 = 0, best_c2 = 0;
  __int128_t max_dist_sq_num = -1;
  __int128_t max_dist_sq_den = 1;

  int k = hull.size();
  for (int i = 0; i < k; ++i) {
    Point p1 = hull[i];
    Point p2 = hull[(i + 1) % k];

    ll a = p2.x - p1.x;
    ll b = p2.y - p1.y;

    ll min_val = LLONG_MAX;
    ll max_val = LLONG_MIN;

    for (const auto &p : points) {
      ll val = a * p.x + b * p.y;
      if (val < min_val) {
        min_val = val;
      }
      if (val > max_val) {
        max_val = val;
      }
    }

    __int128_t current_dist_sq_num =
        (__int128_t)(max_val - min_val) * (max_val - min_val);
    __int128_t current_dist_sq_den = (__int128_t)a * a + (__int128_t)b * b;

    if (current_dist_sq_den == 0)
      continue;

    if (current_dist_sq_num * max_dist_sq_den >
        max_dist_sq_num * current_dist_sq_den) {
      max_dist_sq_num = current_dist_sq_num;
      max_dist_sq_den = current_dist_sq_den;
      best_a = a;
      best_b = b;
      best_c1 = -min_val;
      best_c2 = -max_val;
    }
  }

  std::cout << best_a << " " << best_b << " " << best_c1 << "\n";
  std::cout << best_a << " " << best_b << " " << best_c2 << "\n";

  return 0;
}
