import sys
import math
import random
from sys import stdout

def count_digits(n):
    return len(str(n))

def estimate_cost(n, cost_cache, max_depth=12):
    cost = 0
    current = n
    depth = 0
    
    while current != 1 and depth < max_depth:
        if current % 2 == 0:
            next_val = current // 2
        else:
            next_val = 3 * current + 1
        
        cost += count_digits(next_val)
        current = next_val
        depth += 1
        
        if current in cost_cache and depth < max_depth:
            cost += cost_cache[current]
            break
    
    if depth < max_depth:
        cost_cache[n] = cost
    
    return cost

def analyze_number_properties(n):
    """Analyze statistical properties of the number"""
    score = 0
    
    # Check if it's a power of 2 (very good)
    if n & (n - 1) == 0 and n > 1:
        return 0.1
    
    # Check proximity to powers of 2
    closest_power = 2**int(math.log2(n))
    if abs(n - closest_power) < n * 0.1:
        score += 0.2
    
    # Check number of trailing zeros (factors of 2)
    factors_of_2 = 0
    temp = n
    while temp % 2 == 0:
        factors_of_2 += 1
        temp //= 2
    
    if factors_of_2 >= 3:
        score += 0.3
    elif factors_of_2 >= 1:
        score += 0.1
    
    # Check magnitude (larger numbers tend to be worse)
    magnitude_factor = min(1.0, math.log10(max(1, n)) / 7)
    score += magnitude_factor * 0.4
    
    return min(1.0, score)

def smart_random_jump(n):
    """Intelligent random jump strategy"""
    # Prefer powers of 2 and numbers close to powers of 2
    target_range = range(max(1, n//4), min(n*3, 20_000_000))
    
    # Generate candidates with bias toward good numbers
    candidates = []
    
    # Add powers of 2 in range
    for i in range(1, 25):
        power = 2**i
        if power in target_range:
            candidates.extend([power] * 10)  # Higher weight
    
    # Add numbers close to powers of 2
    for i in range(1, 25):
        power = 2**i
        for offset in [-1, 1, -2, 2, -3, 3]:
            candidate = power + offset
            if candidate in target_range and candidate > 0:
                candidates.extend([candidate] * 3)  # Medium weight
    
    # Add some random numbers
    for _ in range(20):
        candidates.append(random.randint(max(1, n//4), min(n*3, 20_000_000)))
    
    return random.choice(candidates) if candidates else random.randint(max(1, n//4), min(n*3, 20_000_000))

def should_use_random(n, cost_cache, adaptive_threshold):
    """Improved decision making with multiple factors"""
    
    # Factor 1: Immediate cost estimation
    immediate_cost = estimate_cost(n, cost_cache)
    
    # Factor 2: Number properties analysis
    properties_score = analyze_number_properties(n)
    
    # Factor 3: Adaptive threshold
    threshold = adaptive_threshold
    
    # Combined decision score
    decision_score = (
        0.6 * (immediate_cost / threshold) +
        0.4 * properties_score
    )
    
    return decision_score > 0.65

def main():
    cost_cache = {}
    adaptive_threshold = 500
    
    t = int(input())
    
    for _ in range(t):
        n = int(input())
        
        while n != 1:
            if should_use_random(n, cost_cache, adaptive_threshold):
                print("random")
                stdout.flush()
                # Note: In actual implementation, you'd jump here
                # For this simulation, we continue with collatz
                if n % 2 == 0:
                    n = n // 2
                else:
                    n = 3 * n + 1
            else:
                print("collatz")
                stdout.flush()
                if n % 2 == 0:
                    n = n // 2
                else:
                    n = 3 * n + 1
            
            n = int(input())
            if n == 0:
                sys.exit(0)

if __name__ == "__main__":
    main()
