import sys
import math
import random
from sys import stdout
from collections import defaultdict

class AdaptiveThreshold:
    def __init__(self):
        self.threshold = 600
        self.performance_history = []
        self.learning_rate = 0.05
        self.min_threshold = 200
        self.max_threshold = 800
    
    def update_threshold(self, actual_cost, predicted_cost):
        error = actual_cost - predicted_cost
        self.threshold += self.learning_rate * error
        self.threshold = max(self.min_threshold, min(self.max_threshold, self.threshold))

class CollatzOptimizer:
    def __init__(self):
        self.cost_cache = {}
        self.range_performance = defaultdict(list)
        self.threshold_manager = AdaptiveThreshold()
        self.powers_of_2 = {2**i for i in range(1, 25)}
        
    def count_digits(self, n):
        return len(str(n))
    
    def estimate_cost(self, n, max_depth=12):
        cost = 0
        current = n
        depth = 0
        
        while current != 1 and depth < max_depth:
            if current % 2 == 0:
                next_val = current // 2
            else:
                next_val = 3 * current + 1
            
            cost += self.count_digits(next_val)
            current = next_val
            depth += 1
            
            if current in self.cost_cache and depth < max_depth:
                cost += self.cost_cache[current]
                break
        
        if depth < max_depth:
            self.cost_cache[n] = cost
        
        return cost
    
    def expected_collatz_cost(self, n, depth=8):
        total_cost = 0
        current = n
        probability = 1.0
        
        for _ in range(depth):
            if current == 1:
                break
            total_cost += probability * self.count_digits(current)
            if current % 2 == 0:
                current = current // 2
            else:
                current = 3 * current + 1
            probability *= 0.5
        
        return total_cost
    
    def expected_random_cost(self, n):
        return 3 * math.log10(max(1, n)) + 8
    
    def get_range_performance(self, n):
        range_key = n // 1000000  # Group by millions
        if range_key in self.range_performance:
            recent_performance = self.range_performance[range_key][-10:]
            return sum(recent_performance) / len(recent_performance) if recent_performance else 0.5
        return 0.5
    
    def analyze_sequence_properties(self, n):
        # Check if number has properties that suggest good Collatz behavior
        if n in self.powers_of_2:
            return 0.1  # Very good
        
        # Check if number is close to power of 2
        closest_power = 2**int(math.log2(n))
        if abs(n - closest_power) < n * 0.1:
            return 0.2  # Good
        
        # Check if number has many factors of 2
        factors_of_2 = 0
        temp = n
        while temp % 2 == 0:
            factors_of_2 += 1
            temp //= 2
        
        if factors_of_2 >= 3:
            return 0.3  # Decent
        
        return 0.7  # Not great
    
    def should_use_random(self, n):
        immediate_cost = self.estimate_cost(n)
        magnitude_factor = min(1.0, math.log10(max(1, n)) / 7)
        range_performance = self.get_range_performance(n)
        sequence_properties = self.analyze_sequence_properties(n)
        
        # Weighted decision score
        decision_score = (
            0.5 * (immediate_cost / self.threshold_manager.threshold) +
            0.2 * magnitude_factor +
            0.2 * range_performance +
            0.1 * sequence_properties
        )
        
        return decision_score > 0.6
    
    def smart_random_jump(self, n):
        # Try to jump to powers of 2 or numbers close to powers of 2
        target_range = range(max(1, n//4), min(n*3, 10_000_000))
        
        # First, try powers of 2 in range
        candidates = []
        for power in self.powers_of_2:
            if power in target_range:
                candidates.append(power)
        
        if candidates:
            return random.choice(candidates)
        
        # Try numbers close to powers of 2
        for power in self.powers_of_2:
            for offset in [-1, 1, -2, 2]:
                candidate = power + offset
                if candidate in target_range and candidate > 0:
                    candidates.append(candidate)
        
        if candidates:
            return random.choice(candidates)
        
        # Fallback to random in range
        return random.randint(max(1, n//4), min(n*3, 10_000_000))
    
    def update_performance(self, n, actual_cost):
        range_key = n // 1000000
        self.range_performance[range_key].append(actual_cost)
        if len(self.range_performance[range_key]) > 20:
            self.range_performance[range_key] = self.range_performance[range_key][-20:]

def main():
    optimizer = CollatzOptimizer()
    t = int(input())
    
    for _ in range(t):
        n = int(input())
        
        while n != 1:
            if optimizer.should_use_random(n):
                print("random")
                stdout.flush()
                # Simulate random jump
                n = optimizer.smart_random_jump(n)
            else:
                print("collatz")
                stdout.flush()
                # Follow Collatz sequence
                if n % 2 == 0:
                    n = n // 2
                else:
                    n = 3 * n + 1
            
            n = int(input())
            if n == 0:
                sys.exit(0)

if __name__ == "__main__":
    main()
